# 🍪 راه‌حل نهایی: احراز هویت با Cookie + localStorage

## 🎯 مشکل قبلی

Token در localStorage ذخیره می‌شد ولی:
- ❌ PHP نمی‌تونست localStorage رو بخونه
- ❌ وقتی کاربر به صفحه دیگه میرفت، دوباره درخواست لاگین می‌کرد
- ❌ نیاز به redirect بود که تجربه کاربری خوبی نداشت

## ✅ راه‌حل نهایی: Cookie

حالا Token هم در **localStorage** و هم در **Cookie** ذخیره میشه:

### مزایای Cookie:
1. ✅ PHP می‌تونه مستقیماً بخونه (با `$_COOKIE`)
2. ✅ با هر درخواست خودکار ارسال میشه
3. ✅ نیازی به JavaScript یا redirect نیست
4. ✅ در موبایل/WebView عالی کار می‌کنه

## 🔧 تغییرات اعمال شده

### 1. تغییر `config/config.php` - تابع `isLoggedIn()`

```php
function isLoggedIn() {
    // Check session first
    if (isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
        return true;
    }
    
    // Check for token in multiple sources
    $token = null;
    
    // 1. Check Cookie (اولویت اول!) ⭐
    if (isset($_COOKIE['auth_token'])) {
        $token = $_COOKIE['auth_token'];
    }
    
    // 2. Check Authorization header (برای API)
    // 3. Check query parameter (برای لینک‌ها)
    // 4. Check POST parameter (برای فرم‌ها)
    
    if ($token) {
        $user = getUserByToken($token);
        if ($user) {
            // Set session variables
            $_SESSION['user_id'] = $user['id'];
            // ...
            
            // Set cookie if not already set
            if (!isset($_COOKIE['auth_token'])) {
                setcookie('auth_token', $token, time() + (30 * 24 * 60 * 60), '/', '', false, true);
            }
            
            return true;
        }
    }
    
    return false;
}
```

### 2. تغییر `login.php` - ذخیره در Cookie

```javascript
// بعد از لاگین موفق
if (data.token) {
    // ذخیره در localStorage (برای WebView)
    localStorage.setItem('auth_token', data.token);
    localStorage.setItem('user_data', JSON.stringify(data.user));
    
    // ⭐ ذخیره در Cookie (برای PHP)
    document.cookie = 'auth_token=' + data.token + '; path=/; max-age=' + (30*24*60*60) + '; SameSite=Lax';
    
    console.log('✅ لاگین موفق - Token در localStorage و Cookie ذخیره شد');
}
```

### 3. تغییر `logout.php` - پاک کردن Cookie

```php
// Destroy auth token cookie
if (isset($_COOKIE['auth_token'])) {
    setcookie('auth_token', '', time() - 3600, '/', '', false, true);
}
```

```javascript
// پاک کردن از localStorage
localStorage.removeItem('auth_token');
localStorage.removeItem('user_data');
```

### 4. آپدیت `test-token-auth.php`

حالا هم localStorage و هم Cookie رو نمایش میده.

## 🔄 جریان کار

### لاگین:
```
کاربر لاگین می‌کنه
    ↓
Token ساخته میشه
    ↓
Token در localStorage ذخیره میشه  ✅
    ↓
Token در Cookie ذخیره میشه  ✅
    ↓
کاربر به داشبورد میره
```

### دسترسی به صفحات:
```
کاربر روی "eSIM های من" کلیک می‌کنه
    ↓
درخواست به سرور میره
    ↓
PHP خودکار Cookie رو می‌خونه  ✅
    ↓
Token معتبر است؟
    ├─ بله → Session ست میشه → کاربر وارد میشه ✅
    └─ خیر → به /login.php redirect میشه
```

### خروج:
```
کاربر روی "خروج" کلیک می‌کنه
    ↓
Session پاک میشه  ✅
    ↓
Cookie پاک میشه  ✅
    ↓
localStorage پاک میشه  ✅
    ↓
کاربر به صفحه اصلی میره
```

## 🎉 مزایای این روش

1. **سریع**: نیازی به redirect نیست
2. **ساده**: PHP خودکار Cookie رو می‌خونه
3. **ایمن**: 
   - Cookie با HttpOnly flag ست میشه (در PHP)
   - SameSite=Lax برای جلوگیری از CSRF
   - 30 روز اعتبار
4. **سازگار**: در WebView، Browser، و Mobile عالی کار می‌کنه
5. **قابل اعتماد**: هم localStorage هم Cookie، پس اگر یکی fail کرد، دیگری کار می‌کنه

## 🧪 تست

### تست ساده:
1. برو به `/login.php`
2. لاگین کن
3. Console رو باز کن (F12)
4. باید ببینی: `✅ لاگین موفق - Token در localStorage و Cookie ذخیره شد`
5. روی "eSIM های من" کلیک کن
6. **نباید درخواست لاگین مجدد بشه!** ✅

### تست پیشرفته:
1. برو به `/test-token-auth.php`
2. چک کن:
   - ✅ localStorage Token: موجود
   - ✅ Cookie Token: موجود
   - ✅ Session: ست شده
3. در Console بنویس:
```javascript
// چک کردن localStorage
localStorage.getItem('auth_token')

// چک کردن Cookie
document.cookie
```

### تست Cookie در PHP:
برو به `/test-token-auth.php` و بخش "Cookies" رو چک کن.
باید `auth_token` رو ببینی.

## 📝 نکات امنیتی

1. **HttpOnly**: در PHP وقتی `setcookie` صدا میزنیم، flag آخر `true` هست (httponly)
2. **SameSite**: در JavaScript `SameSite=Lax` ست شده
3. **Path**: Cookie فقط برای `/` معتبره
4. **Expiry**: 30 روز اعتبار داره

## 🔧 فایل‌های تغییر یافته

1. ✅ `config/config.php` - خواندن Cookie در `isLoggedIn()`
2. ✅ `login.php` - ذخیره Token در Cookie
3. ✅ `logout.php` - پاک کردن Cookie و localStorage
4. ✅ `test-token-auth.php` - نمایش Cookie

## 🚀 نتیجه

حالا سیستم احراز هویت **کامل و بدون مشکل** کار می‌کنه:
- ✅ Token در localStorage و Cookie ذخیره میشه
- ✅ PHP خودکار Cookie رو می‌خونه
- ✅ Session ست میشه
- ✅ کاربر **هیچ وقت** نیازی به لاگین مجدد نداره
- ✅ در همه دستگاه‌ها عالی کار می‌کنه

---

**تاریخ**: 2025-01-13  
**نسخه**: 3.0 (راه‌حل نهایی با Cookie)

